function coords = getDotsFRS(imageStack,radii,alphaParameter,betaParameter,radiusForThreshold,thresholdParameter,showDots)
%This function finds dots in the image. It could be used as a substitute
%for the wavelet-based method for finding dot; it works very well for
%finding dots in the cells with unhomogenous fluorescence
%
% Input:
%   - imageStack : image stack of the considered movie
%   - radii : vector of integer values at which "gradient" is computed
%   (example can be [1 2 3 4 5])
%   - alphaParameter :  parameter that determines what dots should be
%   emphasized:
%             1 - accepts dots with bilateral symmetry.
%             2 - accepts dots that are between being radially symmetric
%             and vilateral symmetry
%             3 - mostly accepts dots with radial symmetry
%   - betaParameter : gradient threshold; Gradient below this threshold do
%                    not contribute to symmetry measure, defaults value is 0
%   - radiusForThreshold radius of region considered in non-maximal
%                     suppression. Typical values are 1-3 pixels
%   - thresholdParameter - threshold for determining which dots should be
%   found; Typical value can be 3-5
%   - showDots : if you want to see found dots use here value 1, otherwise
%   do not supply this value
%
% Output :
%   - coords :  list of dots [x-coordinates,y-coordinates, frameNR] found in the
%   image
%
% Author: Vladimir Curic, 2015-09-02
%         Vladimir: added frame number to be consistent with the function
%         getDots.m


if nargin < 7
    showDots = 0;
end

% loop over all frames
coords = [];
for ind = 1:size(imageStack, 3)
    
    
    [S, So] = fastradial(imageStack(:,:,ind), radii, alphaParameter, betaParameter);
    
    % VC(&ML): find dots using the map S. One can consider map So; This
    % choice is application dependent. However, our hunch is that S is
    % better for finding dots, while So might be better for
    % localization in some cases (but there we have MLE methods
    % anyway).
    [r,c, rsubp, csubp] = nonmaxsuppts(S, radiusForThreshold, thresholdParameter, imageStack(:,:,ind));
    
    % plot dots on the input image
    if showDots == 1
        figure(2), imshow(imageStack(:,:,ind),[]);
        %subplot(122), imshow(imageStack(:,:,ind),[]);
        hold on
        for kk = 1:length(csubp)
            circle(csubp(kk),rsubp(kk),1)
            hold on
        end
        %pause
    end
    
    coord = horzcat(csubp,rsubp,ind*ones(length(csubp),1));
    coords = [coords;coord];
end

end